/* FILE: mdt28xx.c                              (D. Tottingham  04/07/91)

This is a collection of C functions that manage the DT28xx board for xdetect.
All functions have been written and compiled medium model.  The following
functions are included:

dt_buffer_full ()               return buffer status
dt_dump_configuration ()        dump the DT2821 configuration
dt_get_atodinfo ()              return DT_28xx structure
dt_get_bankswitch_count ()      return bank switch count for this session
dt_get_buffer_size ()           get buffer size
dt_get_channel_size ()          get channel block size
dt_get_channel_gain ()          get channel gain
dt_get_clip_value ()            get clip value
dt_get_clock_source ()          get clock source
dt_get_data_type ()             get data type
dt_get_data_units ()            get data units
dt_get_dc_offset ()             get dc offset
dt_get_digitization_rate ()     get digitization rate
dt_get_input_range ()           get input range
dt_get_new_buffers ()           get new buffer(s) from DT2821
dt_get_next_buffer ()           get next buffer from mux queue
dt_get_number_of_ext_buffers () get number of external buffers
dt_get_scan_count ()            get scan count
dt_get_trigger_source ()        get trigger source
dt_initialize_ADC ()            initialize the DT2821
dt_initialize_buffers ()        initialize buffers
dt_initialize_mux ()            initialize multiplexer
dt_initialize_params()          initialize parameters
dt_set_channel ()               put channel and gain in channel and gain lists
dt_set_ChannelBlocksize ()      set channel block size
dt_set_ChannelGain ()           set the channel gain
dt_set_ClockSource ()           set clock source
dt_set_DigitizationRate ()      set the digitization rate
dt_set_mux_type ()              set the multiplexer type
dt_set_TriggerSource ()         set trigger source
dt_start_ADC ()                 start continuous data acquisition
dt_stop_ADC ()                  stop continuous data acquisition

EXTERNAL FUNCTIONS CALLED:

er_abort ()                     display an error message then quit
h_increment_bankswitch_ctr ()   increment bank switch counter
o_write_logfile ()              write string to log file stream
suds_initialize ()              initialize a suds structure
suds_initialize_tag ()          initialize a suds structtag

HISTORY:
   none

*/



/*************************************************************************
                            INCLUDE FILES

*************************************************************************/
#include <conio.h>
#include <malloc.h>
#include <math.h>
#include <stdio.h>

#include <atldefs.h>                    /* Data Translation includes    */
#include "atlerrs.h"

#include "mconst.h"
#include "mdt28xx.h"
#include "merror.h"
#include "mlog.h"
#include "mqueue.h"
#include "mscrnhdr.h"
#include "mstation.h"
#include "msudsini.h"
#include "mutils.h"
#include "xdetect.h"


/************************************************************************
                                DEFINES

************************************************************************/
/* Multiplexer types */
#define NO_MUX                  0x00ff
#define SE_128x16_V2X           0x0000
#define SE_128x16_V1            0x0001
#define DI_64x4_V1              0x0008

/* Multiplexer commands */
#define NULL_COMMAND            0x0000
#define WRITE_NBANKS            0x0001
#define READ_MUXID              0x0002
#define READ_CURRENT_BANK       0x0003

/* Gain types */
#define NO_GAIN                 0
#define GAIN_1                  1
#define GAIN_1248               2
#define GAIN_500                4

/* DT28xx Constants */
#define BOARD_CLOCK_FREQUENCY   4e6
#define DIO_CONTROL             0x0006
#define DIO_DATA                0x000a
#define DT28XX_UNIT             1
#define HIGH_BYTE_ENABLE        0x0002
#define LOW_BYTE_ENABLE         0x0001
#define WORD_ENABLE             0x0003
#define WORD_DISABLE            0x0000


/*************************************************************************
                                GLOBALS

*************************************************************************/
PRIVATE struct {
        unsigned int mux_id;
        unsigned int mux_factor;
        char *mux_descriptor;
        int adc_device_id;
        FLAG diff_inputs;
} mux_configuration[] = {
        DI_64x4_V1,     16,     "DI_64x4_V1",           (DT2823        |
                                                         DT2827),       YES,

        SE_128x16_V1,   8,      "SE_128x16_V1",         (DT2821        |
                                                         DT2821_F_SE   |
                                                         DT2821_G_SE   |
                                                         DT2824_PGH    |
                                                         DT2824_PGL    |
                                                         DT2825),       NO,

        SE_128x16_V2X,  8,      "SE_128x16_V2X",        (DT2821        |
                                                         DT2821_F_SE   |
                                                         DT2821_G_SE   |
                                                         DT2824_PGH    |
                                                         DT2824_PGL    |
                                                         DT2825),       NO,

        NO_MUX,         1,      NULL,                   UNUSED,         NO
};

PRIVATE struct {
        int adc_device_id;
        unsigned int gain_type;
        unsigned int n_bits;
} dt28xx_info[] = {
        DT2821,         (GAIN_1 | GAIN_1248),           12,
        DT2821_F_SE,    (GAIN_1 | GAIN_1248),           12,
        DT2821_F_DI,    (GAIN_1 | GAIN_1248),           12,
        DT2821_G_SE,    (GAIN_1 | GAIN_1248),           12,
        DT2821_G_DI,    (GAIN_1 | GAIN_1248),           12,
        DT2823,         GAIN_1,                         16,
        DT2824_PGH,     (GAIN_1 | GAIN_1248),           12,
        DT2824_PGL,     (GAIN_1 | GAIN_500),            12,
        DT2825,         (GAIN_1 | GAIN_500),            12,
        DT2827,         GAIN_1,                         16,
        DT2828,         GAIN_1,                         12,
        UNUSED,         NO_GAIN,                        0
};

PRIVATE struct {
        unsigned int max_gain;
        unsigned int gain_type;
} max_gain_map[] = {
        1,              GAIN_1,
        8,              GAIN_1248,
        500,            GAIN_500,
        0,              NO_GAIN,
};

PRIVATE struct {
        unsigned int gain;
        unsigned int gain_type;
} gain_map[] = {
        1,              GAIN_1,
        2,              GAIN_1248,
        4,              GAIN_1248,
        8,              GAIN_1248,
        10,             GAIN_500,
        100,            GAIN_500,
        500,            GAIN_500,
        0,              NO_GAIN
};

PRIVATE DT_28XX adc_info, far * a = &adc_info;
PRIVATE AL_CONFIGURATION config, far * c = &config;
PRIVATE int status;
PRIVATE Q_QUEUE mux_queue;
PRIVATE Q_LINK * next_ptr;
PRIVATE double digitization_rate;
PRIVATE unsigned long buffer_size;
PRIVATE unsigned long channel_blocksize;
PRIVATE unsigned int scan_count;
PRIVATE unsigned int channel_gain;
PRIVATE unsigned int max_channel;
PRIVATE unsigned int bankswitch_count;
PRIVATE unsigned int mux_id;
PRIVATE FLAG mux_installed;


/*=======================================================================*
 *                           get_dynamic_range                           *
 *=======================================================================*/
/* Get the dynamic range from the dt28xx_info table.                     */

PRIVATE
FLAG get_dynamic_range (dr_ptr)
unsigned int *dr_ptr;
{
   int i;

   for (i = 0; dt28xx_info[i].adc_device_id != UNUSED; i++)
      if (dt28xx_info[i].adc_device_id == c->device_id) {
         *dr_ptr = dt28xx_info[i].n_bits;
         return TRUE;
      }

   return FALSE;
}

/*=======================================================================*
 *                              get_max_gain                             *
 *=======================================================================*/
/* Get the maximum gain possible for given device.                       */

PRIVATE
FLAG get_max_gain (mg_ptr)
unsigned int *mg_ptr;
{
   unsigned int max_gain_type;
   int i, j;

   for (i = 0; dt28xx_info[i].adc_device_id != UNUSED &&
               dt28xx_info[i].adc_device_id != c->device_id; i++);

   if (dt28xx_info[i].adc_device_id != UNUSED) {
      max_gain_type = NO_GAIN;
      if (dt28xx_info[i].gain_type & GAIN_500)
         max_gain_type = GAIN_500;
      else if (dt28xx_info[i].gain_type & GAIN_1248)
         max_gain_type = GAIN_1248;
      else if (dt28xx_info[i].gain_type & GAIN_1)
         max_gain_type = GAIN_1;
      for (j = 0; max_gain_map[j].gain_type != max_gain_type; j++);
      *mg_ptr = max_gain_map[j].max_gain;
      return TRUE;
   }

   return FALSE;
}

/*=======================================================================*
 *                           get_mux_descriptor                          *
 *=======================================================================*/
/* Get the multiplexer descriptor for the given mux_id.                  */

PRIVATE
FLAG get_mux_descriptor (mux_id, md_ptr)
unsigned int mux_id;
char ** md_ptr;
{
   int i;

   for (i = 0; mux_configuration[i].mux_id != NO_MUX; i++)
      if (mux_configuration[i].mux_id == mux_id) {
         *md_ptr = mux_configuration[i].mux_descriptor;
         return TRUE;
      }
   *md_ptr = NULL;
   return FALSE;
}

/*=======================================================================*
 *                             get_mux_factor                            *
 *=======================================================================*/
/* Get the multiplex factor for the given mux_id.                        */

PRIVATE
FLAG get_mux_factor (mux_id, mf_ptr)
unsigned int mux_id;
unsigned int *mf_ptr;
{
   int i;

   for (i = 0; mux_configuration[i].mux_id != NO_MUX; i++)
      if (mux_configuration[i].mux_id == mux_id) {
         *mf_ptr = mux_configuration[i].mux_factor;
         return TRUE;
      }
   return FALSE;
}

/*=======================================================================*
 *                           compute_dc_offset                           *
 *=======================================================================*/
/* Compute the dc offset.                                                */

PRIVATE
FLAG compute_dc_offset (dc_ptr)
unsigned int *dc_ptr;
{
   unsigned int i, dr;

   /* Mid-point is a function of the encoding scheme. */
   switch (c->device_flags.encoding) {
      case OFFSET_BINARY:
         if (get_dynamic_range (&dr)) {
            for (i = 1, *dc_ptr = 1; i < dr; i++)
               *dc_ptr <<= 1;
            return TRUE;
         }
         break;

      case STRAIGHT_BINARY:
      case TWOS_COMPLEMENT:
         *dc_ptr = 0;
         return TRUE;
         break;
   }

   return FALSE;
}

/*=======================================================================*
 *                           send_mux_command                            *
 *=======================================================================*/
/* Send the given command and data to multiplexer.                       */

/* Command word, upper nibble definitions */
#define LATCH_2827      0x0080
#define CLEAR_BIT       0x0040
#define LATCH_2821      0x0020

PRIVATE
void send_mux_command (command, data)
unsigned int command, data;
{
   unsigned int command_word;

   command_word = command | CLEAR_BIT | (data << 8);

   /* Send the command, latch = 0 */
   outpw ((c->base_address + DIO_DATA), command_word);

   /* Send the command, latch = 1 */
   outpw ((c->base_address + DIO_DATA), command_word | LATCH_2827);

   /* Send the command, latch = 0
   outpw ((c->base_address + DIO_DATA), command_word);
   */
}

/*=======================================================================*
 *                         allocate_int_buffers                          *
 *=======================================================================*/
/* Allocate "internal" buffer queue for ATLAB.  All internal buffers
   reside in the base 640K of user RAM.                                  */

PRIVATE
void allocate_int_buffers ()
{
   static unsigned int far int_buffer_number;
   unsigned int actual_blocksize, actual_buffersize, temp;
   double seconds_in_buffer;
   unsigned long offset;
   Q_TYPE type;
   Q_BUFFER far * mux;
   char * netwname;

   actual_buffersize = (buffer_size <= MAX_BUFFER_SIZE) ? buffer_size : MAX_BUFFER_SIZE;
   actual_blocksize = actual_buffersize / scan_count;
   seconds_in_buffer = actual_blocksize / digitization_rate;

   q_initialize (&mux_queue);
   netwname = st_get_netwname ();
   for (offset = 0; offset < buffer_size; offset += MAX_BUFFER_SIZE) {
      mux = (Q_BUFFER far *) _fmalloc (sizeof (Q_BUFFER));
      if (mux == NULL) er_abort (DT_NO_STORAGE);

      mux->data = (int far *) _fmalloc ( actual_buffersize * sizeof(unsigned));
      if (mux->data == NULL) er_abort (DT_NO_STORAGE);

      AL_DECLARE_BUFFER ( &int_buffer_number, mux->data, actual_buffersize );

      if (get_dynamic_range (&temp))
         mux->dynamic_range = temp;

      mux->buffer_number = int_buffer_number;
      mux->buffer_size = actual_buffersize;
      mux->seconds_in_buffer = seconds_in_buffer;

      suds_initialize (MUXDATA, &mux->info);
      suds_initialize_tag (MUXDATA, &mux->structtag);

      mux->structtag.len_data = actual_buffersize * sizeof(unsigned);
      u_strncpy (mux->info.netname, ((char far *)netwname), 4);

      if (compute_dc_offset (&temp))
         mux->info.dc_offset = temp;

      mux->info.blocksize = actual_blocksize;
      mux->info.numchans = scan_count;
      mux->info.dig_rate = digitization_rate;
      mux->info.typedata = 's';
      type.buffer = mux;
      q_enqueue (&mux_queue, type);
   }
}

/*=======================================================================*
 *                         allocate_ext_buffers                          *
 *=======================================================================*/
/* Allocate an "external" buffer for ATLAB.  All external buffers reside
   ABOVE the base 640K of user RAM.                                      */

PRIVATE
void allocate_ext_buffers ()
{
   static unsigned far ext_buffer_number;

   a->info.extended_bufs = 0;
   while (AL_ALLOCATE_XM_BUFFER (&ext_buffer_number, buffer_size) == ALE_NORMAL) {
      AL_LINK_BUFFER ( ext_buffer_number );
      a->info.extended_bufs++;
   }
}

/*=======================================================================*
 *                              clear_mux                                *
 *=======================================================================*/
/* Clear the multiplexer.                                                */

PRIVATE
void clear_mux ()
{
   unsigned int command_word;

   outpw ((c->base_address + DIO_CONTROL), LOW_BYTE_ENABLE);

   command_word = (NULL_COMMAND & 0x0f) | LATCH_2821 | LATCH_2827;

   /* Send the NULL_COMMAND, clear = 0 */
   outpw ((c->base_address + DIO_DATA), command_word);

   /* Send the command, clear = 1 */
   outpw ((c->base_address + DIO_DATA), command_word | CLEAR_BIT);
}

/*=======================================================================*
 *                          devflags_to_bitmap                           *
 *=======================================================================*/
/* Convert DEV_FLAGS structure to compact 16-bit bit map.                */

PRIVATE
unsigned short devflags_to_bitmap (devflags)
DEV_FLAGS devflags;
{
   unsigned short bitmap;

   bitmap = ((unsigned short)devflags.DMA_mode) << 14;
   bitmap += ((unsigned short)devflags.SE_DI) << 13;
   bitmap += ((unsigned short)devflags.unipolar) << 12;
   bitmap += ((unsigned short)devflags.encoding) << 10;
   bitmap += ((unsigned short)devflags.sign_extended) << 9;

   return (bitmap);
}

/*=======================================================================*
 *                           read_current_bank                           *
 *=======================================================================*/
/* Read current bank from multiplexer.                                   */

PRIVATE
unsigned int read_current_bank ()
{
   unsigned current_bank;

   outpw ((c->base_address + DIO_CONTROL), LOW_BYTE_ENABLE);
   send_mux_command (READ_CURRENT_BANK | LATCH_2821, 0);
   current_bank = inpw (c->base_address + DIO_DATA);
   return ((current_bank >> 8) & 0x00ff);
}

/*=======================================================================*
 *                              read_muxid                               *
 *=======================================================================*/
/* Read multiplexer id from multiplexer.                                 */

PRIVATE
unsigned int read_muxid()
{
   unsigned int muxid;

   outpw ((c->base_address + DIO_CONTROL), LOW_BYTE_ENABLE);
   send_mux_command (READ_MUXID | LATCH_2821, 0);
   muxid = inpw (c->base_address + DIO_DATA);
   return ((muxid >> 8) & 0x00ff);
}

/*=======================================================================*
 *                             write_nbanks                              *
 *=======================================================================*/
/* Send number of banks to multiplexer.                                  */

PRIVATE
void write_nbanks (nbanks)
unsigned int nbanks;
{
   unsigned int command_word;

   outpw ((c->base_address + DIO_CONTROL), WORD_ENABLE);

   send_mux_command (WRITE_NBANKS, nbanks);
   send_mux_command (WRITE_NBANKS | LATCH_2821, nbanks);
   send_mux_command (NULL_COMMAND | LATCH_2821, nbanks);
}

/*=======================================================================*
 *                               valid_gain                              *
 *=======================================================================*/
/* Determines whether gain is valid in this system.                      */

PRIVATE
FLAG valid_gain (gain)
unsigned int gain;
{
   int i, j;
   unsigned int gain_type;

   if (gain == 0)
      return TRUE;

   /* Translate gain into gain type */
   for (i = 0; gain_map[i].gain_type != NO_GAIN &&
               gain_map[i].gain != gain; i++);

   /* Look for device and gain match in dt28xx_info table */
   if (gain_map[i].gain_type != NO_GAIN) {
      for (j = 0; dt28xx_info[j].adc_device_id != UNUSED; j++)
         if (c->device_id == dt28xx_info[j].adc_device_id) {
            if (gain_map[i].gain_type & dt28xx_info[j].gain_type)
               return TRUE;
            else return FALSE;
         }
   } else return FALSE;
}

/*=======================================================================*
 *                           valid_multiplexer                           *
 *=======================================================================*/
/* Determines whether multiplexer is valid in this system.               */

PRIVATE
FLAG valid_multiplexer (mux_id)
unsigned int mux_id;
{
   int i;

   for (i = 0; mux_configuration[i].mux_id != NO_MUX; i++)
      if (mux_id == mux_configuration[i].mux_id &&
          (c->device_id & mux_configuration[i].adc_device_id)) {
         if ((c->device_flags.SE_DI && mux_configuration[i].diff_inputs) ||
             (!c->device_flags.SE_DI && !mux_configuration[i].diff_inputs))
            return TRUE;
         else return FALSE;
      }

   return FALSE;
}

/*=======================================================================*
 *                              valid_mux_id                             *
 *=======================================================================*/
/* Determines whether mux id is valid.                                   */

PRIVATE
FLAG valid_mux_id (mux_id)
unsigned int mux_id;
{
   int i;

   for (i = 0; mux_configuration[i].mux_id != NO_MUX; i++)
      if (mux_id == mux_configuration[i].mux_id)
         return TRUE;

   return FALSE;
}

/*=======================================================================*
 *                           dt_buffer_full                              *
 *=======================================================================*/
/* Return buffer status.                                                 */

PUBLIC
FLAG dt_buffer_full ()
{
   static unsigned int far buffer_number, far count_remaining;
   int errnum;

   errnum = AL_TEST_BUFFER (&buffer_number, &count_remaining);
   if (errnum == ALE_NOTINPROG)
      er_abort (DT_IO_NOTINPROG);

   return ((count_remaining) ? FALSE : TRUE);
}

/*=======================================================================*
 *                        dt_dump_configuration                          *
 *=======================================================================*/
/* Dump the DT2821 configuration to a stream.                            */

PUBLIC
void dt_dump_configuration (stream)
FILE * stream;
{
   int i;

   if ( status == ALE_NORMAL ) {
      if (c->device_id == UNUSED)
         fprintf(stream, "No device defined.\r\n");
      else {
         fprintf (stream, "Device id is ");
         switch (c->device_id) {
            case DT2828:
                    fprintf(stream, "DT2828");
                    break;
            case DT2827:
                    fprintf(stream, "DT2827");
                    break;
            case DT2821_F_SE:
                    fprintf(stream, "DT2821-F-16SE");
                    break;
            case DT2821_F_DI:
                    fprintf(stream, "DT2821-F-8DI");
                    break;
            case DT2821:
                    fprintf(stream, "DT2821");
                    break;
            case DT2821_G_DI:
                    fprintf(stream, "DT2821_G_DI");
                    break;
            case DT2821_G_SE:
                    fprintf(stream, "DT2821_G_SE");
                    break;
            case DT2823:
                    fprintf(stream, "DT2823");
                    break;
            case DT2825:
                    fprintf(stream, "DT2825");
                    break;
            case DT2824_PGH:
                    fprintf(stream, "DT2824_PGH");
                    break;
            case DT2824_PGL:
                    fprintf(stream, "DT2824_PGL");
                    break;
            case DT2829:
                    fprintf(stream, "DT2829");
                    break;
         }
         fprintf(stream, ", ");

         switch (c->device_flags.DMA_mode) {
            case PROGRAMMED_IO:
                    fprintf(stream, "programmed I/O");
                    break;
            case SINGLE_CHANNEL:
                    fprintf(stream, "single channel DMA");
                    break;
            case DUAL_CHANNEL:
                    fprintf(stream, "dual channel DMA");
                    break;
         }
         fprintf(stream, ", ");

         if (c->device_flags.SE_DI)
            fprintf(stream, "DI, ");
         else
            fprintf(stream, "SE, ");

         if (c->device_flags.unipolar)
            fprintf(stream, "unipolar, ");
         else
            fprintf(stream, "bipolar, ");

         switch (c->device_flags.encoding) {
            case OFFSET_BINARY:
                    fprintf(stream, "offset binary encoding");
                    break;
            case STRAIGHT_BINARY:
                    fprintf(stream, "straight binary encoding");
                    break;
            case TWOS_COMPLEMENT:
                    fprintf(stream, "two's complement encoding");
                    if (c->device_flags.sign_extended)
                       fprintf(stream, ", sign extended");
                    break;
         }
      fprintf(stream, "\r\n");
      fprintf(stream, "Base I/O address is %4x hex.\r\n",c->base_address);
/*
      fprintf(stream, "%d A/D channels.\r\n",c->channel_count);
*/
      }
   }
}

/*=======================================================================*
 *                           dt_get_atodinfo                             *
 *=======================================================================*/
/* Return DT_28xx structure.                                             */

PUBLIC
DT_28XX far * dt_get_atodinfo ()
{
   return (a);
}


/*=======================================================================*
 *                       dt_get_bankswitch_count                         *
 *=======================================================================*/
/* Return bank switch count for this session.                            */

PUBLIC
unsigned int dt_get_bankswitch_count ()
{
   return (bankswitch_count);
}

/*=======================================================================*
 *                          dt_get_buffer_size                           *
 *=======================================================================*/
/* Get buffer size.                                                      */

PUBLIC
unsigned long dt_get_buffer_size ()
{
   return (buffer_size);
}

/*=======================================================================*
 *                         dt_get_channel_gain                           *
 *=======================================================================*/
/* Get channel gain.                                                     */

PUBLIC
unsigned int dt_get_channel_gain (channel)
unsigned int channel;
{
   return ((channel < c->channel_count) ? a->gain_list[channel] : channel_gain);
}

/*=======================================================================*
 *                         dt_get_channel_size                           *
 *=======================================================================*/
/* Get channel block size.                                               */

PUBLIC
unsigned long dt_get_channel_size ()
{
   return (channel_blocksize);
}

/*=======================================================================*
 *                          dt_get_clip_value                            *
 *=======================================================================*/
/* Get the clip value.                                                   */

PUBLIC
unsigned long dt_get_clip_value ()
{
   unsigned int i, dr;
   unsigned long cv;

   cv = 1;
   if (get_dynamic_range (&dr))
      for (i = 0; i < dr; i++)
         cv <<= 1;

   return cv;
}

/*=======================================================================*
 *                           dt_get_clock_source                         *
 *=======================================================================*/
/* Get clock source.                                                     */

PUBLIC
char dt_get_clock_source ()
{
   return (a->info.timing_source);
}

/*=======================================================================*
 *                          dt_get_data_type                             *
 *=======================================================================*/
/* Get the data type.                                                    */

PUBLIC
char dt_get_data_type ()
{
   return 's';
}

/*=======================================================================*
 *                          dt_get_data_units                            *
 *=======================================================================*/
/* Get the data units.                                                   */

PUBLIC
char dt_get_data_units ()
{
   return 'd';
}

/*=======================================================================*
 *                          dt_get_dc_offset                             *
 *=======================================================================*/
/* Get the dc offset.                                                    */

PUBLIC
unsigned int dt_get_dc_offset ()
{
   unsigned int offset;

   compute_dc_offset (&offset);
   return offset;
}

/*=======================================================================*
 *                        dt_get_digitization_rate                       *
 *=======================================================================*/
/* Get digitization rate.                                                */

PUBLIC
double dt_get_digitization_rate ()
{
   return (digitization_rate);
}

/*=======================================================================*
 *                         dt_get_input_range                            *
 *=======================================================================*/
/* Get the input range.                                                  */

PUBLIC
unsigned int dt_get_input_range ()
{
   if (c->device_flags.unipolar)
      return 10;
   else
      return 20;
}

/*=======================================================================*
 *                          dt_get_new_buffers                           *
 *=======================================================================*/
/* Wait for buffer to complete.  Copy external buffer into user
   accessible mux queue.                                                 */

PUBLIC
void dt_get_new_buffers ()
{
   static unsigned int far released_buffer, far unit;
   unsigned long offset;
   FLAG bank_switched;
   int errnum;
   Q_LINK * head;

   /* Wait for a buffer to complete */
   offset = 0;
   unit = DT28XX_UNIT;

   bank_switched = FALSE;
   errnum = AL_RELEASE_BUFFER (&unit, &released_buffer);
   switch (errnum) {
      case ALE_NORMAL:
      case ALE_BANKSWITCH:
         /* Make a scene if we have a bank switch */
         if (mux_installed && mux_id != SE_128x16_V1 &&
             errnum == ALE_BANKSWITCH) {
            o_write_logfile ("WARNING: Bank switch occured on multiplexer\n");
            if (Debug_enabled)
               printf ("WARNING: Bank switch occured on multiplexer\n");
            bankswitch_count++;
/** this is very ugly **/
            h_increment_bankswitch_ctr();
            bank_switched = TRUE;
         }

         head = next_ptr = mux_queue.head;
         while (head != NULL) {
            AL_COPY_BUFFER (released_buffer, head->type.buffer->buffer_number, offset,
                            head->type.buffer->buffer_size);
            offset += head->type.buffer->buffer_size;
            head->type.buffer->bank_switched = bank_switched;
            head = head->next;
         }
         AL_RETURN_BUFFER (released_buffer);

         break;

      case ALE_INPTMO:
         er_abort (DT_TIMEOUT);
         break;

      case ALE_BUFFER:
      default:
         er_abort (DT_OVERFLOW);
         break;
   }
}

/*=======================================================================*
 *                          dt_get_next_buffer                           *
 *=======================================================================*/
/* Get next buffer from mux queue.                                       */

PUBLIC
Q_BUFFER far * dt_get_next_buffer ()
{
   Q_LINK * this_ptr;

   this_ptr = next_ptr;
   if (next_ptr != NULL)
      next_ptr = next_ptr->next;

   return (this_ptr->type.buffer);
}

/*=======================================================================*
 *                      dt_get_number_of_ext_buffers                     *
 *=======================================================================*/
/* Get number of external buffers.                                       */

PUBLIC
int dt_get_number_of_ext_buffers ()
{
   return (a->info.extended_bufs);
}

/*=======================================================================*
 *                            dt_get_scan_count                          *
 *=======================================================================*/
/* Get scan count.                                                       */

PUBLIC
int dt_get_scan_count ()
{
   return (scan_count);
}

/*=======================================================================*
 *                           dt_get_trigger_source                       *
 *=======================================================================*/
/* Get trigger source.                                                   */

PUBLIC
char dt_get_trigger_source ()
{
   return (a->info.trigger_source);
}

/*=======================================================================*
 *                           dt_initialize_ADC                           *
 *=======================================================================*/
/* Initialize the DT28xx for ADC.                                        */

PUBLIC
void dt_initialize_ADC ()
{
   static long far number_of_ticks;
   unsigned int actual_scancount;
   char out_str[80], *mx_descript;
   int errnum;

   /* Do some error checking */
   if (!scan_count)
      er_abort (DT_NO_STATIONS);
   else if (! u_ispow2( (unsigned long)scan_count))
      er_abort (DT_INVALID_SCANCOUNT);

   actual_scancount = (scan_count > c->channel_count) ? c->channel_count :
                      scan_count;

   /* Reset the device */
   AL_RESET ();

   /* Give ATLAB the channel and gain lists */
   AL_SETUP_ADC ( ((a->info.trigger_source == 'i') ? INTERNAL_TRIGGER : EXTERNAL_TRIGGER) +
                  ((a->info.timing_source == 'i') ? INTERNAL_CLOCK : EXTERNAL_CLOCK),
                  actual_scancount, a->channel_list, a->gain_list);

   /* Compute and send number of banks to multiplexer */
if (Debug_enabled)
   printf ("mux id is %u\n", mux_id);

   if (mux_installed) {
      switch (mux_id) {
         case SE_128x16_V1:
            /* Set the external mux bits */
            a->info.external_mux = scan_count / c->channel_count;
            outpw ((a->info.base_address + DIO_DATA), a->info.external_mux);
            outpw ((a->info.base_address + DIO_CONTROL), LOW_BYTE_ENABLE);
            break;

         default:
            a->info.external_mux = (scan_count - 1) / c->channel_count;
            write_nbanks (a->info.external_mux);
      }

/**debug**/
if (Debug_enabled) {
   printf ("Mux bits = 0x000%x\n", a->info.external_mux);
   printf ("Latched mux bits...\n");
}
   }

   /* Send a description of the current mux to the screen and to the log file */
   if (get_mux_descriptor (mux_id, &mx_descript)) {
      sprintf (out_str, "%s multiplexer installed.\n", mx_descript);
      o_write_logfile (out_str);
      printf ("%s", out_str);
   } else {
      sprintf (out_str, "No multiplexer installed.\n");
      o_write_logfile (out_str);
      printf ("%s", out_str);
   }

   /* Write the number of channels being digitized to screen and to log file */
   sprintf (out_str, "%3d channels defined\n", scan_count);
   o_write_logfile (out_str);
   printf ("%s", out_str);

   /* Set the sampling rate */
   number_of_ticks = (long) (BOARD_CLOCK_FREQUENCY /
                     (digitization_rate * scan_count));
   if ((errnum = AL_SET_CLOCK (number_of_ticks)) != ALE_NORMAL) {
      if (errnum == ALE_SMALLP) er_abort (DT_FREQTOOHIGH);
      else er_abort (DT_FREQTOOLOW);
   }

   if (a->info.timing_source == 'i') {
      AL_RETURN_CLOCK (&number_of_ticks);
      digitization_rate = (double)(BOARD_CLOCK_FREQUENCY) / number_of_ticks /
                           scan_count;
   }

/**debug**/
if (Debug_enabled)
   printf ("Channel digitization rate = %10.4lf\n", digitization_rate);

   /* Set the timeout rate */
   AL_SET_TIMEOUT ( TIME_OUT * (unsigned int) (((double)channel_blocksize) /
                    digitization_rate));
}

/*=======================================================================*
 *                           dt_initialize_buffers                       *
 *=======================================================================*/
/* Initialize buffers.                                                   */

PUBLIC
void dt_initialize_buffers ()
{
   buffer_size = channel_blocksize * scan_count;
   buffer_size = (! buffer_size) ? SEGMENT_SIZE : buffer_size;
   channel_blocksize = buffer_size / scan_count;

   /* Do some error checking first */
   if (! u_ispow2( channel_blocksize))
      er_abort (DT_INVALID_CHANNELSIZE);
   else if (buffer_size > SEGMENT_SIZE)
      er_abort (DT_INVALID_BUFFERSIZE);

   allocate_int_buffers ();
   allocate_ext_buffers ();
}


/*=======================================================================*
 *                           dt_initialize_mux                           *
 *=======================================================================*/
/* Initialize multiplexer.                                               */

PUBLIC
void dt_initialize_mux()
{
   mux_installed = FALSE;
}

/*=======================================================================*
 *                          dt_initialize_params                         *
 *=======================================================================*/
/* Initialize parameters.                                                */

PUBLIC
void dt_initialize_params()
{
   unsigned int mux_factor, max_gain;

   /* Set error processing */
   AL_SET_ERROR_CONTROL_WORD (Debug_enabled);

   /* Open a channel to the ATLAB device driver */
   if (AL_INITIALIZE () != ALE_NORMAL) er_abort (DT_NO_ATLDRV);

   /* Address board DT28XX_UNIT, the first unit */
   if (AL_SELECT_BOARD ( DT28XX_UNIT ) != ALE_NORMAL) er_abort (DT_NO_BOARD);

   /* Get the current configuration */
   status = AL_GET_CONFIGURATION (c);

   /* Initialize the multiplexer */
   bankswitch_count = 0;
   clear_mux();
   if (! mux_installed) {

      /* Is there a mux installed? Get multiplexer id. */
      if ((mux_id = read_muxid()) != NO_MUX)
         mux_installed = TRUE;
      else mux_installed = FALSE;

      /* Multiplexer is installed. Is it a valid mux type? */
      if (mux_installed && !valid_multiplexer (mux_id))
         er_abort (DT_INVALID_MUX);
   }

   /* Initialize max_channel */
   max_channel = c->channel_count;
   if (mux_installed && get_mux_factor (mux_id, &mux_factor))
      max_channel *= mux_factor;

   /* Set channel gain based on max gain and channel gain default */
   channel_gain = CHANNEL_GAIN;
   if (!valid_gain (channel_gain)) {
      if (get_max_gain (&max_gain))
         channel_gain = max_gain;
      else er_abort (DT_INVALID_GAIN);
   }

   digitization_rate = DIGITIZATION_RATE;
   channel_blocksize = 0;
   scan_count = 0;

   suds_initialize (ATODINFO, &a->info);
   suds_initialize_tag (ATODINFO, &a->structtag);

   if (status == ALE_NORMAL) {
      a->info.base_address = c->base_address;
      a->info.device_id = c->device_id;
      a->info.device_flags = devflags_to_bitmap (c->device_flags);
   }
   a->info.trigger_source = 'i';
   a->info.timing_source = 'i';
   a->info.external_mux = 0;

   a->channel_list = (int far *) _fmalloc (c->channel_count * sizeof(int));
   if (a->channel_list == NULL) er_abort (DT_NO_STORAGE);

   a->gain_list = (int far *) _fmalloc (c->channel_count * sizeof(int));
   if (a->gain_list == NULL) er_abort (DT_NO_STORAGE);
}

/*=======================================================================*
 *                              dt_set_channel                           *
 *=======================================================================*/
/* Put channel number in channel list and gain number in gain list.      */

PUBLIC
FLAG dt_set_channel (channel, gain)
unsigned int channel;
unsigned int gain;
{
   char out_str[80];

   if (channel >= max_channel) {
      sprintf (out_str, "WARNING: Invalid channel %3d ignored.\n", channel);
      o_write_logfile (out_str);
      printf ("%s", out_str);
      return FALSE;
   }

   if (channel != scan_count)
      er_abort (DT_INVALID_CHANNEL);

   if (! mux_installed) {
      if (!valid_gain (gain))
         er_abort (DT_INVALID_GAIN);

      a->channel_list[scan_count] = channel;
      a->gain_list[scan_count] = (gain == 0) ? channel_gain : gain;
   } else if (scan_count < c->channel_count) {
      a->channel_list[scan_count] = channel;
      a->gain_list[scan_count] = channel_gain;
   }
   scan_count++;
   return TRUE;
}

/*=======================================================================*
 *                         dt_set_ChannelBlocksize                       *
 *=======================================================================*/
/* Set the channel block size.                                           */

PUBLIC
void dt_set_ChannelBlocksize (blocksize)
unsigned long blocksize;
{
   channel_blocksize = blocksize;
}

/*=======================================================================*
 *                            dt_set_ChannelGain                         *
 *=======================================================================*/
/* Set the channel gain.                                                 */

PUBLIC
void dt_set_ChannelGain (gain)
unsigned int gain;
{
   int i;

   if (!valid_gain (gain))
      er_abort (DT_INVALID_GAIN);
   channel_gain = gain;
   for (i = 0; i < scan_count && i < c->channel_count; i++)
      a->gain_list[i] = gain;
}

/*=======================================================================*
 *                            dt_set_ClockSource                         *
 *=======================================================================*/
/* Set clock source.                                                     */

PUBLIC
void dt_set_ClockSource (timing_source)
int timing_source;
{
   a->info.timing_source = (timing_source == INTERNAL_CLOCK) ? 'i' : 'e';
}

/*=======================================================================*
 *                          dt_set_DigitizationRate                      *
 *=======================================================================*/
/* Set the digitization rate.                                            */

PUBLIC
void dt_set_DigitizationRate (rate)
double rate;
{
   digitization_rate = rate;
}

/*=======================================================================*
 *                             dt_set_mux_type                           *
 *=======================================================================*/
/* Set the multiplexer type.                                             */

PUBLIC
void dt_set_mux_type (type)
unsigned int type;
{
   mux_installed = TRUE;
   mux_id = type;
   if (mux_id == NO_MUX)
      mux_id = SE_128x16_V1;

   if (!valid_mux_id (mux_id))
      er_abort (DT_INVALID_MUX);
}

/*=======================================================================*
 *                           dt_set_TriggerSource                        *
 *=======================================================================*/
/* Set trigger source.                                                   */

PUBLIC
void dt_set_TriggerSource (trigger_source)
int trigger_source;
{
   a->info.trigger_source = (trigger_source == INTERNAL_TRIGGER) ? 'i' : 'e';
}

/*=======================================================================*
 *                            dt_start_ADC                               *
 *=======================================================================*/
/* Start continuous data acquisition.                                    */

PUBLIC
void dt_start_ADC ()
{
   /* Clear the multiplexer */
   if (mux_installed && mux_id != SE_128x16_V1)
      clear_mux();

   AL_CONTINUOUS_ADC ();
}

/*=======================================================================*
 *                            dt_stop_ADC                                *
 *=======================================================================*/
/* Stop continuous data acquisition.  Reset the DT28xx.                  */

PUBLIC
void dt_stop_ADC ()
{
   AL_STOP ();
   AL_RESET ();
   AL_TERMINATE ();
}
